# 機能設計書 14-HTTP Basic認証

## 概要

本ドキュメントはFastAPIフレームワークにおけるHTTP Basic認証機能の設計仕様を定義する。

### 本機能の処理概要

HTTP Basic認証機能は、RFC 7617で定義されたHTTPベーシック認証スキームを実装する。HTTPBasicクラスをDependsまたはSecurity依存関係として使用することで、エンドポイントにBasic認証を適用できる。認証情報（ユーザー名とパスワード）はBase64でエンコードされてAuthorizationヘッダーで送信される。

**業務上の目的・背景**：HTTP Basic認証は、最も基本的なHTTP認証方式であり、シンプルな認証要件を持つAPIやサービスで広く使用されている。セットアップが容易で、ほとんどのHTTPクライアントが標準でサポートしているため、内部API、開発環境、管理者向けエンドポイントなどで利用される。

**機能の利用シーン**：
- 管理者向けAPIの認証
- 開発環境でのシンプルな認証
- サードパーティシステムとの連携（Basic認証が要求される場合）
- 内部サービス間の認証
- Swagger UIでの認証テスト

**主要な処理内容**：
1. HTTPリクエストからAuthorizationヘッダーを取得
2. Basicスキームの検証
3. Base64デコードによりユーザー名とパスワードを分離
4. HTTPBasicCredentialsオブジェクトとして返却
5. 認証失敗時のHTTPException発生（auto_error=Trueの場合）

**関連システム・外部連携**：
- OpenAPI: HTTPセキュリティスキームの定義
- Swagger UI: Basic認証のテスト

**権限による制御**：Basic認証自体は権限制御を行わない。ユーザー名に基づくロールベースアクセス制御は開発者が実装する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Swagger UI | 補助機能 | HTTP Basic認証のテスト。ユーザー名とパスワードの入力フォーム表示 |

## 機能種別

セキュリティ / 認証

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| scheme_name | str \| None | No | セキュリティスキーム名 | 文字列 |
| realm | str \| None | No | HTTP Basic認証のレルム | 文字列 |
| description | str \| None | No | セキュリティスキームの説明 | 文字列 |
| auto_error | bool | No | 認証失敗時に自動でエラーを発生させるか（デフォルト: True） | ブール値 |

### 入力データソース

- HTTPリクエストのAuthorizationヘッダー（`Basic <base64_encoded_credentials>`）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| HTTPBasicCredentials | Pydantic BaseModel | ユーザー名とパスワードを含むオブジェクト |
| credentials.username | str | 認証ユーザー名 |
| credentials.password | str | 認証パスワード |

### 出力先

- パスオペレーション関数のパラメータ
- OpenAPIスキーマ（セキュリティスキーム定義）

## 処理フロー

### 処理シーケンス

```
1. リクエスト受信
   └─ Authorization: Basic <base64> ヘッダーの確認
2. ヘッダー解析
   └─ get_authorization_scheme_param関数でスキームとパラメータを分離
3. スキーム検証
   └─ "basic"（大文字小文字区別なし）であることを確認
4. Base64デコード
   └─ パラメータ部分をBase64デコード（ASCII）
5. クレデンシャル分離
   └─ コロン（:）で区切ってユーザー名とパスワードを分離
6. 結果返却
   └─ HTTPBasicCredentialsオブジェクトを生成して返却
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト受信] --> B{Authorizationヘッダーあり?}
    B -->|No| C{auto_error?}
    B -->|Yes| D[スキームとパラメータを分離]
    D --> E{スキームはbasic?}
    E -->|No| C
    E -->|Yes| F[Base64デコード]
    F --> G{デコード成功?}
    G -->|No| H[HTTPException 401発生]
    G -->|Yes| I[コロンで分割]
    I --> J{セパレータあり?}
    J -->|No| H
    J -->|Yes| K[HTTPBasicCredentials生成]
    K --> L[返却]
    C -->|True| H
    C -->|False| M[None返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-14-01 | Basicスキーム検証 | Authorizationヘッダーのスキームは"basic"（大文字小文字区別なし）である必要がある | HTTPBasic使用時 |
| BR-14-02 | Base64デコード | クレデンシャル部分はBase64でデコードされる | 認証情報の解析時 |
| BR-14-03 | コロン区切り | デコードされた文字列は最初のコロン（:）でユーザー名とパスワードに分離される | クレデンシャル分離時 |
| BR-14-04 | レルム指定 | realmパラメータが指定された場合、WWW-Authenticateヘッダーにレルムを含める | エラーレスポンス時 |
| BR-14-05 | 空パスワード許可 | パスワードは空文字列でも許可される（コロンより前がユーザー名、後がパスワード） | クレデンシャル分離時 |

### 計算ロジック

クレデンシャルの分離:
```python
data = b64decode(param).decode("ascii")
username, separator, password = data.partition(":")
# partitionは最初のコロンで分割し、コロンがない場合はseparatorが空
```

## データベース操作仕様

本機能はデータベース操作を直接行わない。ユーザー認証の検証は開発者が実装する。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| HTTP 401 | Unauthorized | Authorizationヘッダーがない | 正しい認証情報を送信 |
| HTTP 401 | Unauthorized | スキームがbasicでない | Basic形式で送信 |
| HTTP 401 | Unauthorized | Base64デコードに失敗 | 正しくエンコードされた値を送信 |
| HTTP 401 | Unauthorized | コロン区切りがない | `username:password`形式で送信 |

### リトライ仕様

本機能では自動リトライは行わない。

## トランザクション仕様

本機能はトランザクション管理を行わない。

## パフォーマンス要件

- Base64デコードと文字列分割のみの軽量処理
- 各リクエストで認証処理が実行される

## セキュリティ考慮事項

- Basic認証はBase64エンコードであり暗号化ではない。必ずHTTPS経由で使用すること
- パスワードはBase64デコードすると平文で取得されるため、比較時にはハッシュ化されたパスワードと適切に比較すること
- タイミング攻撃を防ぐため、secrets.compare_digestを使用した比較を推奨
- ブルートフォース攻撃対策としてレート制限を実装すること
- realmを指定することで、認証領域を明示できる

## 備考

- HTTPBasicCredentialsはPydantic BaseModelを継承しており、自動的にバリデーションが行われる
- RFC 7617に準拠した実装

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、認証情報を表すデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | http.py | `fastapi/security/http.py` | HTTPBasicCredentialsクラスの定義（16-27行目） |
| 1-2 | models.py | `fastapi/openapi/models.py` | HTTPBaseモデルの定義（348-351行目） |

**読解のコツ**: `HTTPBasicCredentials`はPydantic BaseModelを継承し、`username`と`password`フィールドを持つ。

#### Step 2: 基底クラスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | http.py | `fastapi/security/http.py` | HTTPBaseクラスの定義（69-104行目） |

**主要処理フロー**:
1. **70-82行目**: __init__メソッドでHTTPBaseModelを生成
2. **84-85行目**: make_authenticate_headersメソッドでWWW-Authenticateヘッダー生成
3. **87-92行目**: make_not_authenticated_errorメソッドでエラー生成
4. **94-104行目**: __call__メソッドで認証情報を抽出

#### Step 3: HTTPBasicクラスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | http.py | `fastapi/security/http.py` | HTTPBasicクラスの定義（107-221行目） |

**主要処理フロー**:
- **142-197行目**: __init__メソッドでrealmパラメータを処理
- **194行目**: HTTPBaseModel(scheme="basic")を設定
- **199-202行目**: make_authenticate_headersメソッドでレルムを含める
- **204-221行目**: __call__メソッドでBase64デコードとクレデンシャル分離

#### Step 4: エラー処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | http.py | `fastapi/security/http.py` | __call__メソッド内のエラー処理（214-220行目） |

**主要処理フロー**:
- **214-217行目**: Base64デコード失敗時のエラー発生
- **218-220行目**: コロン区切りがない場合のエラー発生

### プログラム呼び出し階層図

```
HTTPBasic(scheme_name, realm, description, auto_error)
    │
    └─ HTTPBase.__init__()（暗黙的）
           │
           └─ __call__(request)
                  │
                  ├─ request.headers.get("Authorization")
                  │
                  ├─ get_authorization_scheme_param()
                  │      └─ スキームとパラメータ分離
                  │
                  ├─ スキーム検証（"basic"）
                  │
                  ├─ b64decode(param).decode("ascii")
                  │      └─ Base64デコード
                  │
                  └─ data.partition(":")
                         └─ ユーザー名とパスワード分離
                                │
                                └─ HTTPBasicCredentials(username, password)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

HTTPリクエスト          HTTPBasic                   HTTPBasicCredentials
Authorization:    ───▶   __call__()          ───▶   {username, password}
Basic dXNlcjpwYXNz          │
                            ▼
                   get_authorization_scheme_param()
                   スキームとパラメータ分離
                            │
                            ▼
                   b64decode().decode("ascii")
                   "user:pass" 復元
                            │
                            ▼
                   partition(":")
                   ユーザー名とパスワード分離
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| http.py | `fastapi/security/http.py` | ソース | HTTPBasicクラスとHTTPBasicCredentialsの定義 |
| base.py | `fastapi/security/base.py` | ソース | SecurityBaseクラスの定義 |
| models.py | `fastapi/openapi/models.py` | ソース | HTTPBaseモデルの定義 |
| utils.py | `fastapi/security/utils.py` | ソース | get_authorization_scheme_param関数 |
| exceptions.py | `fastapi/exceptions.py` | ソース | HTTPException定義 |
